package gov.va.med.mhv.usermgmt.service.impl;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.service.AbstractService;
import org.tigris.atlas.service.ServiceResponse;

import gov.va.med.mhv.usermgmt.messages.EmployeeManagementMessages;
import gov.va.med.mhv.usermgmt.service.EmployeeSearchResultCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.EmployeeSearchResultServiceResponse;
import gov.va.med.mhv.usermgmt.service.EmployeeSearchService;
import gov.va.med.mhv.usermgmt.service.adapter.ActiveDirectoryEnvironment;
import gov.va.med.mhv.usermgmt.transfer.EmployeeSearchResult;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;

/**
 * Service implementation class for the EmployeeSearch service
 */
public class EmployeeSearchServiceImpl extends AbstractService implements
    EmployeeSearchService
{

    private static final Log LOG = LogFactory.getLog(
        EmployeeSearchServiceImpl.class);

    private LdapContext ldapContext;

    /**
     * Execute the SearchByUserName service
     * @return EmployeeSearchResultServiceResponse
     */
    public EmployeeSearchResultServiceResponse searchByUserName(String userName)
    {
        EmployeeSearchResultServiceResponse response = 
            new EmployeeSearchResultServiceResponse();

        if (userName != null) {
            userName = userName.replaceAll("\\*", ""); // no user entered
                                                        // wildcard characters
            userName = userName.trim(); // trim spaces
        }

        // validate criteria, userName is required
        if (StringUtils.isBlank(userName)) {
            addError(response,
                EmployeeManagementMessages.SEARCH_INVALID_CRITERIA);
            return response;
        }

        // create the search string
        String searchFilter = "(&(objectClass=user)(sAMAccountName=" + userName
            + "))";

        // perform search
        return (EmployeeSearchResultServiceResponse) performSearch(response,
            searchFilter, false);
    }

    /**
     * Execute the SearchByName service
     * @return EmployeeSearchResultCollectionServiceResponse
     */
    public EmployeeSearchResultCollectionServiceResponse searchByName(
        String firstName, String lastName)
    {
        EmployeeSearchResultCollectionServiceResponse response = 
            new EmployeeSearchResultCollectionServiceResponse();

        if (lastName != null) {
            lastName = lastName.replaceAll("\\*", ""); // no user entered
                                                        // wildcard characters
            lastName = lastName.trim(); // trim spaces
        }

        if (firstName != null) {
            firstName = firstName.replaceAll("\\*", ""); // no user entered
                                                            // wildcard
                                                            // characters
            firstName = firstName.trim(); // trim spaces
        } else {
            firstName = "";
        }

        // validate criteria, lastname is required & must be at least 3
        // characters
        if (StringUtils.isBlank(lastName) || lastName.length() < 3) {
            addError(response,
                EmployeeManagementMessages.SEARCH_INVALID_CRITERIA);
            return response;
        }

        // create the search string, add wildcard to end of criteria per
        // requirements
        String searchFilter = "(&(objectClass=user)(sn=" + lastName
            + "*)(givenName=" + firstName + "*))";

        // perform search
        return (EmployeeSearchResultCollectionServiceResponse) performSearch(
            response, searchFilter, true);
    }

    private ServiceResponse performSearch(ServiceResponse response,
        String searchFilter, boolean collectionResponse)
    {
        try {
            ldapContext = new InitialLdapContext(ActiveDirectoryEnvironment
                .getLdapEnvironment(), null);

            NamingEnumeration searchResults = ldapContext.search(
                ActiveDirectoryEnvironment.getSearchBase(), searchFilter,
                ActiveDirectoryEnvironment.getSearchControls());
            // no results found
            if (searchResults.hasMoreElements() == false) {
                addInfo(response, EmployeeManagementMessages.SEARCH_NO_RESULTS);
            }

            while (searchResults.hasMoreElements()) {
                SearchResult searchResult = (SearchResult) searchResults.next();
                Attributes searchResultAttributes = searchResult
                    .getAttributes();
                if (searchResultAttributes != null) {
                    // if we are using this helper method to populate results
                    // for collection response, we add search result into
                    // response in different manner
                    if (collectionResponse) {
                        ((EmployeeSearchResultCollectionServiceResponse) 
                            response).addEmployeeSearchResult(
                            convertToEmployeeSearchResult(searchResultAttributes));
                    } else {
                        ((EmployeeSearchResultServiceResponse) response).
                            setEmployeeSearchResult(
                            convertToEmployeeSearchResult(searchResultAttributes));
                    }
                }
            }
        } catch (NamingException e) {
            LOG.error("Failed to connect to Active Directory", e);
            addError(response, EmployeeManagementMessages.SEARCH_FAILED);
        } finally {
            try {
                if (ldapContext != null) {
                    ldapContext.close(); // attempt to close connection
                                            // regardless
                }
            } catch (NamingException e) {
                LOG.error("Failed to close connection to Active Directory", e);
                addError(response, EmployeeManagementMessages.SEARCH_FAILED);
            } finally {
                ldapContext = null; // release handle to connection regardless
            }
        }
        return response;
    }

    private static final EmployeeSearchResult convertToEmployeeSearchResult(
        Attributes attributes) throws NamingException
    {
        EmployeeSearchResult employeeSearchResult = TransferObjectFactory
            .createEmployeeSearchResult();

        if (attributes.get("sAMAccountName") != null)
            employeeSearchResult.setUserName((String) attributes.get(
                "sAMAccountName").get());

        if (attributes.get("givenName") != null)
            employeeSearchResult.setFirstName((String) attributes.get(
                "givenName").get());

        if (attributes.get("sn") != null)
            employeeSearchResult.setLastName((String) attributes.get("sn")
                .get());

        if (attributes.get("title") != null)
            employeeSearchResult.setTitle((String) attributes.get("title")
                .get());

        if (attributes.get("physicalDeliveryOfficeName") != null)
            employeeSearchResult.setOffice((String) attributes.get(
                "physicalDeliveryOfficeName").get());

        if (attributes.get("department") != null)
            employeeSearchResult.setDepartment((String) attributes.get(
                "department").get());

        if (attributes.get("company") != null)
            employeeSearchResult.setCompany((String) attributes.get("company")
                .get());

        if (attributes.get("telephoneNumber") != null)
            employeeSearchResult.setPhone((String) attributes.get(
                "telephoneNumber").get());

        if (attributes.get("mail") != null)
            employeeSearchResult
                .setEmail((String) attributes.get("mail").get());

        employeeSearchResult.setFullName(employeeSearchResult.getLastName()
            + ", " + employeeSearchResult.getFirstName());

        return employeeSearchResult;
    }
}